<?php

namespace App\Http\Controllers;

use App\DomainRegisters\Register;
use Validator;
use Carbon\Carbon;
use App\Models\Page;
use App\Models\Product;
use App\Models\Frontend;
use App\Models\Language;
use App\Models\Subscriber;
use App\Models\DomainSetup;
use Illuminate\Http\Request;
use App\Models\SupportTicket;
use App\Models\SupportMessage;
use App\Models\ServiceCategory;
use App\Models\AdminNotification;
use App\Models\DomainRegister;
use Illuminate\Support\Facades\Cookie;

class SiteController extends Controller {

    public function index() {
        $reference = @$_GET['reference'];
        if ($reference) {
            session()->put('reference', $reference);
        }
        $pageTitle = 'Home';
        $sections = Page::where('tempname', $this->activeTemplate)->where('slug', '/')->first();
        return view($this->activeTemplate . 'home', compact('pageTitle', 'sections'));
    }

    public function pages($slug) {
        $page = Page::where('tempname', $this->activeTemplate)->where('slug', $slug)->firstOrFail();
        $pageTitle = $page->name;
        $sections = $page->secs;
        return view($this->activeTemplate . 'pages', compact('pageTitle', 'sections'));
    }


    public function contact() {
        $pageTitle = "Contact";
        $sections = Page::where('tempname', $this->activeTemplate)->where('slug', 'contact')->first();
        return view($this->activeTemplate . 'contact', compact('pageTitle', 'sections'));
    }


    public function contactSubmit(Request $request) {
        $this->validate($request, [
            'name' => 'required',
            'email' => 'required',
            'subject' => 'required|string|max:255',
            'message' => 'required',
        ]);

        if (!verifyCaptcha()) {
            $notify[] = ['error', 'Invalid captcha provided'];
            return back()->withNotify($notify);
        }

        $request->session()->regenerateToken();

        $random = getNumber();

        $ticket = new SupportTicket();
        $ticket->user_id = auth()->id() ?? 0;
        $ticket->name = $request->name;
        $ticket->email = $request->email;
        $ticket->priority = 2;


        $ticket->ticket = $random;
        $ticket->subject = $request->subject;
        $ticket->last_reply = Carbon::now();
        $ticket->status = 0;
        $ticket->save();

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = auth()->user() ? auth()->user()->id : 0;
        $adminNotification->title = 'A new support ticket has opened ';
        $adminNotification->click_url = urlPath('admin.ticket.view', $ticket->id);
        $adminNotification->save();

        $message = new SupportMessage();
        $message->support_ticket_id = $ticket->id;
        $message->message = $request->message;
        $message->save();

        $notify[] = ['success', 'Ticket created successfully!'];

        return to_route('ticket.view', [$ticket->ticket])->withNotify($notify);
    }

    public function policyPages($slug, $id) {
        $policy = Frontend::where('id', $id)->where('data_keys', 'policy_pages.element')->firstOrFail();
        $pageTitle = $policy->data_values->title;
        return view($this->activeTemplate . 'policy', compact('policy', 'pageTitle'));
    }

    public function changeLanguage($lang = null) {
        $language = Language::where('code', $lang)->first();
        if (!$language) $lang = 'en';
        session()->put('lang', $lang);
        return back();
    }

    public function cookieAccept() {
        $general = gs();
        Cookie::queue('gdpr_cookie', $general->site_name, 43200);
    }

    public function cookiePolicy() {
        $pageTitle = 'Cookie Policy';
        $cookie = Frontend::where('data_keys', 'cookie.data')->first();
        return view($this->activeTemplate . 'cookie', compact('pageTitle', 'cookie'));
    }

    public function placeholderImage($size = null) {
        $imgWidth = explode('x', $size)[0];
        $imgHeight = explode('x', $size)[1];
        $text = $imgWidth . '×' . $imgHeight;
        $fontFile = realpath('assets/font/RobotoMono-Regular.ttf');
        $fontSize = round(($imgWidth - 50) / 8);
        if ($fontSize <= 9) {
            $fontSize = 9;
        }
        if ($imgHeight < 100 && $fontSize > 30) {
            $fontSize = 30;
        }

        $image     = imagecreatetruecolor($imgWidth, $imgHeight);
        $colorFill = imagecolorallocate($image, 100, 100, 100);
        $bgFill    = imagecolorallocate($image, 175, 175, 175);
        imagefill($image, 0, 0, $bgFill);
        $textBox = imagettfbbox($fontSize, 0, $fontFile, $text);
        $textWidth  = abs($textBox[4] - $textBox[0]);
        $textHeight = abs($textBox[5] - $textBox[1]);
        $textX      = ($imgWidth - $textWidth) / 2;
        $textY      = ($imgHeight + $textHeight) / 2;
        header('Content-Type: image/jpeg');
        imagettftext($image, $fontSize, 0, $textX, $textY, $colorFill, $fontFile, $text);
        imagejpeg($image);
        imagedestroy($image);
    }

    public function maintenance() {
        $pageTitle = 'Maintenance Mode';
        $general = gs();
        if ($general->maintenance_mode == 0) {
            return to_route('home');
        }
        $maintenance = Frontend::where('data_keys', 'maintenance.data')->first();
        return view($this->activeTemplate . 'maintenance', compact('pageTitle', 'maintenance'));
    }

    public function subscribe(Request $request) {

        $validator = Validator::make($request->all(), [
            'email' => 'required|email|max:255|unique:subscribers,email'
        ]);

        if (!$validator->passes()) {
            return response()->json(['error' => $validator->errors()->all()]);
        }

        $newSubscriber = new Subscriber();
        $newSubscriber->email = $request->email;
        $newSubscriber->save();

        return response()->json(['success' => true, 'message' => 'Thank you, we will notice you our latest news']);
    }

    public function blogs() {
        $pageTitle = 'Announcements';
        $sections = Page::where('tempname', $this->activeTemplate)->where('slug', 'announcements')->first();
        return view($this->activeTemplate . 'blogs', compact('pageTitle', 'sections'));
    }

    public function blogDetails($slug, $id) {
        $blog = Frontend::where('id', $id)->where('data_keys', 'blog.element')->firstOrFail();
        $pageTitle = $blog->data_values->title;
        return view($this->activeTemplate . 'blog_details', compact('blog', 'pageTitle'));
    }

    public function serviceCategory($slug = null) {

        $serviceCategory = ServiceCategory::active()->when($slug, function ($category) use ($slug) {
            $category->where('slug', $slug);
        })->firstOrFail();

        $pageTitle = $serviceCategory->name;
        return view($this->activeTemplate . 'service_category', compact('pageTitle', 'serviceCategory'));
    }

    public function productConfigure($categorySlug, $productSlug, $id) {

        $product = Product::active()->where('id', $id)->whereHas('serviceCategory', function ($category) {
            $category->active($category);
        })->whereHas('price', function ($price) {
            $price->filter($price);
        })
            ->with('getConfigs.activeGroup.activeOptions.activeSubOptions.getOnlyPrice')
            ->firstOrFail();

        $domains = [];
        $pageTitle = 'Product Configure';

        if ($product->domain_register) {
            $domains = DomainSetup::active()->orderBy('id', 'DESC')->with('pricing')->get();
        }

        return view($this->activeTemplate . 'product_configure', compact('product', 'pageTitle', 'domains'));
    }

    public function registerDomain(Request $request) {

        $pageTitle = 'Register New Domain';
        $availableDomain = false;
        $getResponse = [];
        $unsupportedTld = null;

        if ($request->all()) {
            
            setTimeLimit();

            $defaultDomainRegister = DomainRegister::getDefault();
            if (!$defaultDomainRegister) {
                $notify[] = ['info', 'There is no default domain register, please setup default domain register'];
                return redirect()->route('register.domain')->withNotify($notify);
            }

            $input = $request->domain;
            $onlyDomain = $this->getDomain($input);
            $tld = $this->getTld($input);

            if (!$input) {
                $notify[] = ['error', 'The domain field is required'];
                return redirect()->route('register.domain')->withNotify($notify);
            }

            $general = gs();
            $domainSetups = DomainSetup::active()->with('pricing')->get(['id', 'extension']);

            if($tld){  
                $matchTld = $domainSetups->where('extension', $tld)->first();
                
                if(!$matchTld){
                    $unsupportedTld = $tld;
                }
            }
         
            try {
                $register = new Register($defaultDomainRegister->alias); //The Register is a class
                $register->command = 'searchDomain';
                $register->domain = $onlyDomain;
                $register->tld = $domainSetups->pluck('extension')->toArray();
                $execute = $register->run();

                if (!$execute['success']) {
                    $notify[] = ['error', $execute['message']];
                    return redirect()->route('register.domain')->withNotify($notify);
                }

                if (@$execute['data']['status'] == 'ERROR') {
                    $notify[] = ['error', $execute['data']['message']];
                    return redirect()->route('register.domain')->withNotify($notify);
                }


                if($execute['regster_name'] == 'namecheap'){ 
                    
                    foreach(@$execute['data']['CommandResponse']['DomainCheckResult'] ?? [] as $index => $responseData){

                        $responseData['@attributes']['Available'];
                        
                        if(!$tld){
                            $request->merge([
                                'noTld'=>true,
                                'domain_without_tld'=>$responseData['@attributes']['Domain'],
                            ]);
                        }

                        if($responseData['@attributes']['Domain'] == $input && $responseData['@attributes']['Available'] == true){
                            $availableDomain = true;
                        }   
                        
                        $domainExtension = $domainSetups->where('extension', $this->getTld($responseData['@attributes']['Domain']))->first();

                        if($responseData['@attributes']['Available'] == true){
                            $getResponse[] = [
                                'domain' => $responseData['@attributes']['Domain'],
                                'extension' => $domainExtension->extension,
                                'available' => true,
                                'domain_setup_id' => $domainExtension->id,
                                'price' => $domainExtension->pricing->firstPrice['price'] ?? 0,
                            ];
                        }else{
                            $getResponse[] = [
                                'domain' => $responseData['@attributes']['Domain'],
                                'extension' => $domainExtension->extension,
                                'available' => false,
                                'domain_setup_id' => $domainExtension->id,
                                'price' => $domainExtension->pricing->firstPrice['price'] ?? 0,
                            ];
                        }

                        if($responseData['@attributes']['Domain'] == $input){  
                            $getResponse[$index] += ['match'=>999];
                        }

                    }

                }else{
                    $i = 0;

                    foreach(@$execute['data'] ?? [] as $index => $responseData){

                        if(!$tld){
                            $request->merge([
                                'noTld'=>true,
                                'domain_without_tld'=>$index,
                            ]);
                        }

                        if($index == $input && $responseData['status'] == 'available'){
                            $availableDomain = true;
                        }   
                    
                        $domainExtension = $domainSetups->where('extension', $this->getTld($index))->first();

                        if($responseData['status'] == 'available'){
                            $getResponse[] = [
                                'domain' => $index,
                                'extension' => $domainExtension->extension,
                                'available' => true,
                                'domain_setup_id' => $domainExtension->id,
                                'price' => $domainExtension->pricing->firstPrice['price'] ?? 0,
                            ];
                        }else{
                            $getResponse[] = [
                                'domain' => $index,
                                'extension' => $domainExtension->extension,
                                'available' => false,
                                'domain_setup_id' => $domainExtension->id,
                                'price' => $domainExtension->pricing->firstPrice['price'] ?? 0,
                            ];
                        }

                        if($index == $input){
                            $getResponse[$i] += ['match'=>999];
                        }

                        $i++;
                    }

                }

            } catch (\Exception $error) {
                $notify[] = ['error', 'Something went wrong'];
                return redirect()->route('register.domain')->withNotify($notify);
            }
            
        }

        return view($this->activeTemplate . 'register_domain', compact('pageTitle', 'getResponse', 'availableDomain', 'unsupportedTld'));
    }

    protected function getDomain($url) {  
        preg_match('@^(https?://)?([a-z0-9_-]+\.)*([a-z0-9_-]+)\.[a-z]{2,6}(/.*)?$@i', $url, $match);
        $domain = @$match[3]; 

        if(!$domain){
            $domain = $url;
        }

        return $domain;
    }

    protected function getTld($url) {
        $tld = parse_url($url); 
        $tld = (isset($tld['scheme']) ? $tld['host'] : $tld['path']);
        $tld = explode('.', $tld);  
        
        if(count($tld) > 1){
            $tld = '.'.end($tld); 
        }else{
            $tld = null;
        }

        return $tld;
    }

    protected function validDomain($url) {
        if (!preg_match('/((http|https|ftp|ftps)\:\/\/)?[a-zA-Z0-9\-\.]+\.[a-zA-Z]{2,4}(\/\S*)?/', $url)) {
            return false;
        }

        return true;
    }

    public function searchDomain(Request $request) {

        setTimeLimit();

        $defaultDomainRegister = DomainRegister::getDefault();
        if (!$defaultDomainRegister) {
            return ['success' => false, 'message' => 'There is no default domain register, Please setup default domain register'];
        }

        $domain = $request->domainName;
        $onlyDomain = $this->getDomain($domain);

        $domainSetups = DomainSetup::active()->with('pricing')->get(['id', 'extension']);

        $register = new Register($defaultDomainRegister->alias); //The Register is a class
        $register->command = 'searchDomain';
        $register->domain = $onlyDomain; 
        $register->tld = $domainSetups->pluck('extension')->toArray();
        $execute = $register->run();

        if (!$execute['success']) {
            return ['success' => false, 'message' => $execute['message']];
        }

        if (@$execute['data']['status'] == 'ERROR') {
            return ['success' => false, 'message' => $execute['data']['message']];
        }
  
        return ['success' =>true, 'regster_name'=>$execute['regster_name'], 'data' => $execute['data']];
    }

}
